function [order,wn] = ellipord(wp,ws,rp,rs,opt)
%ELLIPORD Elliptic filter order selection.
%	[N, Wn] = ELLIPORD(Wp, Ws, Rp, Rs) returns the order N of the
%	lowest order digital elliptic filter that loses no more than
%	Rp dB in the passband and has at least Rs dB of attenuation
%	in the stopband.  The passband runs from 0 to Wp and the 
%	stopband extends from Ws to 1.0, the Nyquist frequency.  
%	ELLIPORD also returns Wn, the elliptic natural frequency to
%	use with ELLIP to achieve the specifications.
%	[N, Wn] = ELLIPORD(Wp, Ws, Rp, Rs, 's') does the computation
%	for an analog filter.
%	See also ELLIP, BUTTORD, CHEB1ORD, and CHEB2ORD.

%	Author(s): L. Shure, 6-9-88
%		   T. Krauss, 11-18-92, updated
%	Copyright (c) 1984-94 by The MathWorks, Inc.
%	$Revision: 1.6 $  $Date: 1994/01/25 17:59:04 $

%	Reference(s):
%	    [1] Rabiner and Gold, p 241.

if nargin == 4
	opt = 'z';
elseif nargin == 5
	if ~strcmp(opt,'z') & ~strcmp(opt,'s')
		error('Invalid option for final argument.');
	end
end
np1 = length(wp);
ns1 = length(ws);
if (np1 ~= ns1)
	error('The frequency vectors must both be the same length.')
end
ftype = 2*(np1 - 1);
if wp(1) < ws(1)
	ftype = ftype + 1;	% low (1) or reject (3)
else
	ftype = ftype + 2;	% high (2) or pass (4)
end

% natural frequencies are simply the given band-edges:
wn=wp;

% first, prewarp frequencies from digital (unit circle) to analog (imag. axis):
if strcmp(opt,'z')	% digital
	WP=tan(pi*wp/2);
	WS=tan(pi*ws/2);
else  % don't have to if analog already
	WP=wp;
	WS=ws;
end

% next, transform to low pass prototype with passband edge of 1 and stopband
% edges determined by the following: (see Rabiner and Gold, p.258)
if ftype == 1	% low
	WA=WS/WP;
elseif ftype == 2	% high
	WA=WP/WS;
elseif ftype == 3	% stop
	WA=(WS*(WP(1)-WP(2)))./(WS.^2 - WP(1)*WP(2));
elseif ftype == 4	% pass
	WA=(WS.^2 - WP(1)*WP(2))./(WS*(WP(1)-WP(2)));
end

% find the minimum order elliptic filter to meet the more demanding spec:
WA = min(abs(WA));
epsilon = sqrt(10 .^ (0.1*rp)-1);
k1 = epsilon./sqrt(10 .^ (0.1*rs)-1);
k = 1/WA;
capk = ellipke([k.^2 1-k.^2]);
capk1 = ellipke([(k1 .^2) 1-(k1 .^2)]);
order = ceil(capk(1)*capk1(2)/(capk(2)*capk1(1)));

