function [num, den, z, p] = cheby1(n, r, Wn, ftype)
%CHEBY1	Chebyshev type I digital filter design.
%	[B,A] = CHEBY1(N,R,Wn) designs an N'th order lowpass digital
%	Chebyshev filter with R decibels of ripple in the passband.
%	CHEBY1 returns the filter coefficients in length N+1 vectors B
%	and A.  The cut-off frequency Wn must be 0.0 < Wn < 1.0, 
%	with 1.0 corresponding to half the sample rate.  Use R=0.5 as
%	a starting point, if you are unsure about choosing R.
%
%	If Wn is a two-element vector, Wn = [W1 W2], CHEBY1 returns an 
%	order 2N bandpass filter with passband  W1 < W < W2.
%	[B,A] = CHEBY1(N,R,Wn,'high') designs a highpass filter.
%	[B,A] = CHEBY1(N,R,Wn,'stop') is a bandstop filter if Wn = [W1 W2].
%
%	When used with three left-hand arguments, as in
%	[Z,P,K] = CHEBY1(...), the zeros and poles are returned in
%	length N column vectors Z and P, and the gain in scalar K. 
%
%	When used with four left-hand arguments, as in
%	[A,B,C,D] = CHEBY1(...), state-space matrices are returned. 
%
%	See also CHEB1ORD, CHEBY2, BUTTER, FREQZ and FILTER.

%	J.N. Little 1-14-87
%	Revised 1-13-88 JNL, 4-29-88 LS.
%	Copyright (c) 1987-91 by the MathWorks, Inc.

%	References:
%	  [1] T. W. Parks and C. S. Burrus, Digital Filter Design,
%	      John Wiley & Sons, 1987, chapter 7, section 7.3.3.

btype = 1;
if nargin == 4
	btype = 3;
end
if max(size(Wn)) == 2
	btype = btype + 1;
end

% step 1: get analog, pre-warped frequencies
fs = 2;
u = 2*fs*tan(pi*Wn/fs);

% step 2: convert to low-pass prototype estimate
if btype == 1	% lowpass
	Wn = u;
elseif btype == 2	% bandpass
	Bw = u(2) - u(1);
	Wn = sqrt(u(1)*u(2));	% center frequency
elseif btype == 3	% highpass
	Wn = u;
elseif btype == 4	% bandstop
	Bw = u(2) - u(1);
	Wn = sqrt(u(1)*u(2));	% center frequency
end

% step 3: Get N-th order Chebyshev type-I lowpass analog prototype
[z,p,k] = cheb1ap(n, r);

% Transform to state-space
[a,b,c,d] = zp2ss(z,p,k);

% step 4: Transform to lowpass, bandpass, highpass, or bandstop of desired Wn
if btype == 1		% Lowpass
	[a,b,c,d] = lp2lp(a,b,c,d,Wn);

elseif btype == 2	% Bandpass
	[a,b,c,d] = lp2bp(a,b,c,d,Wn,Bw);

elseif btype == 3	% Highpass
	[a,b,c,d] = lp2hp(a,b,c,d,Wn);

elseif btype == 4	% Bandstop
	[a,b,c,d] = lp2bs(a,b,c,d,Wn,Bw);
end

% step5: Use Bilinear transformation to find discrete equivalent:
[a,b,c,d] = bilinear(a,b,c,d,fs);

if nargout == 4
	num = a;
	den = b;
	z = c;
	p = d;
else	% nargout <= 3
% Transform to zero-pole-gain and polynomial forms:
	if nargout == 3
		[z,p,k] = ss2zp(a,b,c,d,1);
		num = z;
		den = p;
		z = k;
	else % nargout <= 2
		den = poly(a);
		num = poly(a-b*c)+(d-1)*den;
	end
end

