function [yi,weights]=interp2(x,y,xi,weights)
%INTERP2 1-D biharmonic data interpolation.
%	Given data vectors X and Y, and a new abscissa vector XI, the
%	function YI = INTERP2(X,Y,XI) uses biharmonic interpolation
%	to find a vector YI corresponding to XI.
%
%	Here's an example that generates a coarse sine curve, then
%	interpolates over a finer abscissa:
%
%		x = 0:10;  y = sin(x);
%		xi = 0:.25:10;
%		yi = interp2(x,y,xi);
%		plot(x,y,'o',xi,yi)
%
%	[YI,WEIGHTS]=INTERP2(X,Y,XI) also returns the interpolation
%	WEIGHTS, which can be reused as follows:
%	INTERP2(X,Y,XI,WEIGHTS) uses the given WEIGHTS vector instead
%	of recalculating them from X and Y.
%	See also SPLINE and INTERP1.

%	INTERP2 called with no arguments will interpolate and
%	plot some random (x,y) values.

% Reference:  David T. Sandwell, Biharmonic spline
%  interpolation of GEOS-3 and SEASAT altimeter
%  data, Geophysical Research Letters, 2, 139-142,
%  1987.  Describes interpolation using value or
%  gradient of value in any dimension.

% Charles R. Denham, MathWorks, 1988, 1989.

% Copyright (c) The MathWorks, Inc., 1988, 1989.

if nargin < 1         % Data for demonstration plot.
   rand('uniform')
   x = rand(10,1);
   y = rand(10,1);
   x(1) = 0;
   x(2) = 1;
   xi = (0:100) ./ 100;
end

x = x(:);
y = y(:);
m = length(x);

if nargin < 4
   xm = x * ones(1,m);
   d = abs(xm - xm.');
   g = d .^ 3;   % Green's function.
   weights = g \ y;
end

xi = xi(:);
[im,in] = size(xi);
xm = ones(im,1) * x.';
xim = xi * ones(1,m);
d = abs(xim - xm);
g = d .^ 3;   % Green's function.
yi = g * weights;

if nargin < 1        % Demonstration plot.
   hold off
   plot(xi,yi,'-'), hold on
   plot(x,y,'*'), hold off
   title('Demonstration of INTERP2')
   xlabel('x'), ylabel('y')
end
