function [x0,y0] = fplot(fname,lims,npts,angl,subdiv)
%FPLOT	Plot function.
%	FPLOT(FNAME,LIMS) plots the function specified by the text variable
%	FNAME between the x-axis limits specified by LIMS = [XMIN XMAX].
%	For example, FPLOT('sin',[0 10]) is a plot of the sine function.
%	FPLOT(FNAME,LIMS,N) plots the function with a minimum of N samples.
%	The default value for N is 25.
%	FPLOT(FNAME,LIMS,N,ANGLE) plots the function with a minimum of N
%	samples and samples more if contiguous segments are at an angle
%	larger than ANGLE (in degrees).  The default for ANGLE is 10 degrees.
%	FPLOT(FNAME,LIMS,N,ANGLE,SUBDIV) plots the function with a minimum
%	of N samples, no more than ANGLE degrees bends in the curve, but no
%	more than SUBDIV iterations to fill in the rapidly changing portions.
%	The default for SUBDIV is 20.
%	[X,Y] = fplot(FNAME,LIMS,...) returns the abscissae and ordinates for
%	FNAME in the column vectors X and Y.  Plot X versus Y to see the picture.

%	L. Shure, 12-22-88
%	(c) Copyright 1988, by The MathWorks, Inc.

if nargin < 2
	error('The X-axis limits must be specified for FPLOT.');
elseif nargin < 5
	subdiv = 20;
	if nargin < 4
		angl = 10;
	end
	if nargin < 3
		npts = 25;
	end
end
iter = 0;
angl = angl*pi/180;
angl = abs(angl);
ang = angl + 1;
xmin = min(lims);
xmax = max(lims);
x = xmin + (0:npts-1)'*(xmax-xmin)/(npts-1);
y = feval(fname,x);
yscal = max(abs(y));
y = y/yscal;

% see if need to fill in values for 'smoothness'
while any(ang > angl) & (iter <= subdiv)
	yi = diff(y);
	xi = diff(x);
	nx = length(xi) - 1;
	radi = sqrt(xi.^2+yi.^2);
	num = xi(1:nx).*xi(2:nx+1) + yi(1:nx).*yi(2:nx+1);
	den = radi(1:nx).*radi(2:nx+1);
	ang = abs(acos(num./den));
	iter = iter + 1;
	ii = find(ang > angl);	% need to interpolate more
	if isempty(ii)
		break;
	else
		xt = (x(ii) + x(ii+1))/2;
		yt = feval(fname,xt)/yscal;
		[x, ind] = sort([x;xt]);
		y = [y;yt];
		y(:) = y(ind);
	end
end
y = y*yscal;
if iter >= subdiv
	disp('WARNING: Maximum number of iterations has been reached.');
	if nargout == 0
		disp('         Plot may be inaccurate.');
	end
end
if nargout == 0
	plot(x,y)
else
	x0 = x;
	y0 = y;
end
