function clabel(cs, contours)
%CLABEL	Add contour labels to a contour plot.
%	CLABEL(CS) adds height labels to the current contour plot
%	using the contour structure CS output from the CONTOUR routine.
%	The label positions are selected randomly.  For example:
%
%	   cs = contour(rand(10)); clabel(cs)
%
%	CLABEL(CS,V) labels just those contour levels given in
%	vector V.  The default action is to label all known contours.
%
%	CLABEL(CS,'manual') places contour labels at the locations
%	clicked on with a mouse.  Pressing the return key terminates
%	labeling.  Use the space bar to enter contours and the arrow
%	keys to move the crosshair if no mouse is available.
%
%	CLABEL with no arguments contours and labels some random
%	data.  Then, it enters manual mode for additional hand
%	labeling.  The purpose is demonstrational.
%
%	See also CONTOUR and GINPUT.

% Charles R. Denham, MathWorks, 1988, 1989.
% Copyright (c) The MathWorks, Inc., 1989.

% Revised 4/12/90 by CRD: XYLIST now in screen coordinates.  More reliable.
% Revised 4/16/90 by CRD: Removed an inappropriate test.

hold off
rand('uniform')

if nargin < 2, contours = 0; end

choice = 0; manual = 0;
if nargin > 1
   if isstr(contours)
      manual = strcmp(contours, 'manual');
     else
      choice = 1;
   end
end

% Setup for demonstration.

demo = 0;
if nargin < 1
   a = fix(rand(5,5) .* 100);
   contours = 10:10:90;
   cs = contour(a, contours);
   if isempty(cs)
      error('Requires a later version of Matlab.')
   end
   contours = 20:20:80;
   choice = 1; demo = 1;
   disp(' '), disp('    Demonstration of CLABEL.')
end

[mcs, ncs] = size(cs);
contours = sort(contours(:));

% Decompose contour data structure if manual mode.

if manual
   disp(' '), disp('    Please wait a moment...')
   x = []; y = []; clist = []; k = 0; n = 0;
   while (1)
      k = k + n + 1; if k > ncs, break, end
      c = cs(1,k); n = cs(2,k); nn = 2 .* n -1;
      xtemp = zeros(nn, 1); ytemp = zeros(nn, 1);
      xtemp(1:2:nn) = cs(1, k+1:k+n);
      xtemp(2:2:nn) = (xtemp(1:2:nn-2) + xtemp(3:2:nn)) ./ 2;
      ytemp(1:2:nn) = cs(2, k+1:k+n);
      ytemp(2:2:nn) = (ytemp(1:2:nn-2) + ytemp(3:2:nn)) ./ 2;
      x = [x; xtemp]; y = [y; ytemp];   % Keep these.
      clist = [clist; c .* ones(2*n-1, 1)];
   end
   ax = axis; ax = axis;   % Do two times to preserve AXIS state.
   xmin = ax(1); xmax = ax(2); ymin = ax(3); ymax = ax(4);
   xrange = xmax - xmin; yrange = ymax - ymin;
   xylist = (x .* yrange + sqrt(-1) .* y .* xrange);
   disp(' ');
   disp('   Carefully select contours for labeling.')
   disp('   When done, press RETURN or click outside the graph.')
end

k = 0; n = 0; flip = 0;

ENT = 3;   % Ascii for "ENTER" key.
CR = 13;   % Ascii for "RETURN" key.

while (1)

% Use GINPUT and select nearest point if manual.

   if manual
      [xx, yy, button] = ginput(1);
      if button == ENT | button == CR, break, end
      if xx < xmin | xx > xmax, break, end
      if yy < ymin | yy > ymax, break, end
      xy = xx .* yrange + sqrt(-1) .* yy .* xrange;
      dist = abs(xylist - xy);
      f = find(dist == min(dist));
      if length(f) > 0
         f = f(1); xx = x(f); yy = y(f); c = clist(f);
         okay = 1;
        else
         okay = 0;
      end
   end

% Select a labeling point randomly if not manual.

   if ~manual
      k = k + n + 1; if k > ncs, break, end
      c = cs(1, k); n = cs(2, k);
      if choice
         f = find(c == contours);
         okay = length(f) > 0;
        else
         okay = 1;
      end
      if okay
         j = fix(rand .* (n - 1)) + 1;
         if flip, j = n - j; end
         flip = ~flip;
         x1 = cs(1, j+k); y1 = cs(2, j+k);
         x2 = cs(1, j+k+1); y2 = cs(2, j+k+1);
         xx = (x1 + x2) ./ 2; yy = (y1 + y2) ./ 2;   % Test was here; removed.
      end
   end

% Label the point.

   if okay
      s = sprintf('%0.3g', c);
      hold on
      polymark(xx, yy, '+'), text(xx, yy, s),
      hold off
   end

end

% Recursive call if demo.

if demo, clabel(cs, 'manual'), end
