%GAUSSM Mixture of Gaussians density estimate
%
%  W = GAUSSM(A,K)
%  W = A*GAUSSM([],K);
%
% INPUT
%   A	  Dataset
%   K   Number of Gaussians to use (default: 1)
%
% OUTPUT
%   W   Mixture of Gaussians density estimate
%
% DESCRIPTION
% Estimation of a PDF for the dataset A by a Mixture of Gaussians
% procedure. Use is made of EMCLUST(A,QDC,K). Unlabeled objects are
% neglected, unless A is entirely unlabeled or double. Then all objects
% are used. If A is a multi-class crisp labeled dataset the densities are
% estimated class by class and then weighted and combined according their
% prior probabilities. In all cases, just single density estimator W is
% computed.
%
% Note that it is necessary to set the label type of A to soft labels
% (A = LABTYPE(A,'soft') in order to use the traditional EM algorithm
% based on posterior probabilities instead of using crisp labels.
% See also EMCLUST.
% 
% The mapping W may be applied to a new dataset B using DENSITY = B*W.
%
% SEE ALSO
% DATASETS, MAPPINGS, QDC, MOGC, EMCLUST, PLOTM, TESTC

% Copyright: R.P.W. Duin, r.p.w.duin@prtools.org
% Faculty EWI, Delft University of Technology
% P.O. Box 5031, 2600 GA Delft, The Netherlands

% $Id: gaussm.m,v 1.12 2006/03/07 16:17:44 duin Exp $

function w = gaussm(a,n)

	prtrace(mfilename)

  if (nargin < 2)
		prwarning (2,'number of Gaussians not specified, assuming 1.');
		n = 1; 
	end

	% No arguments specified: return an empty mapping.

	if (nargin < 1) | (isempty(a))
		w = mapping(mfilename,n);
		w = setname(w,'Mixture of Gaussians');
		return
	end
	
	if (~isdataset(a) | (getsize(a,3) ~= 1 & islabtype(a,'crisp')))
		w = mclassm(a,mapping(mfilename,n),'weight');
		return
	end

	lablist = getlablist(a);
	[m,k] = getsize(a);
	
	if n == 1
		p = getprior(a);
		[U,G] = meancov(a);
		res.mean = +U;
		res.cov  = G;
		res.prior= 1;
	else
		[e,v] = emclust(a,qdc,n);	
		ncomp0 = size(v.data.mean,1);	
		iter = 0;
		while (ncomp0 ~= n & iter < 5)   % repeat until exactly n components are found
	 		[e,v1] = emclust(a,qdc,n);				
 			ncomp1 = size(v1.data.mean,1);
			if ncomp1 > ncomp0
				v = v1;
				ncomp0 = ncomp1;
			end
			iter = iter + 1;
  		end
  	res.mean = v.data.mean;
  	res.cov  = v.data.cov;
  	res.prior= v.data.prior;
		res.nlab = ones(n,1); % defines that all Gaussian components have to be
		                      % combined into a single class.
	end
	w = mapping('normal_map','trained',res,lablist,k,1);
	w = setname(w,'Mixture of Gaussians');

return
